<?php

namespace App\Controllers;

use App\Models\PenjualanBukuModel;
use App\Models\BukuModel;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Dompdf\Dompdf;

class PendapatanBukuController extends BaseController
{
    protected $penjualanBukuModel;
    protected $bukuModel;

    public function __construct()
    {
        $this->penjualanBukuModel = new PenjualanBukuModel();
        $this->bukuModel = new BukuModel();
    }

    public function index()
{
    // Ambil data dari request (filter tanggal jika ada)
    $tanggal_mulai = $this->request->getGet('tanggal_mulai');
    $tanggal_selesai = $this->request->getGet('tanggal_selesai');

    // Jika tidak ada filter tanggal, set tanggal default
    if (!$tanggal_mulai) {
        $tanggal_mulai = date('Y-m-01'); // Awal bulan
    }
    if (!$tanggal_selesai) {
        $tanggal_selesai = date('Y-m-t'); // Akhir bulan
    }

    // Ambil data penjualan berdasarkan filter tanggal
    $penjualan = $this->penjualanBukuModel
        ->where('tanggal_penjualan >=', $tanggal_mulai)
        ->where('tanggal_penjualan <=', $tanggal_selesai)
        ->findAll();

    // Ambil semua ID buku dari penjualan
    $bukuIds = array_column($penjualan, 'id_buku'); // Ambil semua ID buku dari penjualan

    // Pastikan ada ID buku sebelum menjalankan query
    $bukuData = [];
    if (!empty($bukuIds)) {
        $bukuData = $this->bukuModel->whereIn('id_buku', $bukuIds)->findAll(); // Ambil data buku berdasarkan ID
    }

    // Buat array untuk memetakan ID buku ke nama buku
    $bukuMap = [];
    foreach ($bukuData as $buku) {
        $bukuMap[$buku['id_buku']] = $buku['nama_buku'];
    }

    // Gabungkan nama buku ke data penjualan
    foreach ($penjualan as &$item) {
        $item['nama_buku'] = $bukuMap[$item['id_buku']] ?? 'Tidak Ditemukan'; // Menambahkan nama buku ke penjualan
    }

    // Kirim data ke view
    $data = [
        'penjualan' => $penjualan,
        'tanggal_mulai' => $tanggal_mulai,
        'tanggal_selesai' => $tanggal_selesai
    ];

    return view('dashboard/penjualan/index', $data);
}

    

    public function create()
    {
        $data['buku'] = $this->bukuModel->findAll(); // Mengambil semua buku
        return view('dashboard/penjualan/create', $data);
    }

    public function store()
    {
        $input = $this->request->getPost();

        // Mengambil data buku berdasarkan ID
        $buku = $this->bukuModel->find($input['id_buku']);

        // Pastikan buku ditemukan sebelum melanjutkan
        if (is_null($buku)) {
            return redirect()->back()->with('error', 'Buku tidak ditemukan.');
        }

        // Pastikan stok cukup sebelum melakukan penjualan
        if ($buku['stok'] < $input['jumlah_terjual']) {
            return redirect()->back()->with('error', 'Stok tidak cukup untuk buku: ' . $buku['nama_buku']);
        }

        // Hitung total harga
        $total_harga = $input['jumlah_terjual'] * $buku['harga']; // Mengakses harga sebagai indeks array

        $data = [
            'id_buku' => $input['id_buku'],
            'jumlah_terjual' => $input['jumlah_terjual'],
            'total_harga' => $total_harga,
            'tanggal_penjualan' => date('Y-m-d'), // Menggunakan tanggal hari ini
            'keterangan' => $input['keterangan'],
        ];

        // Simpan data penjualan
        $this->penjualanBukuModel->save($data);

        // Update stok buku
        $this->bukuModel->update($input['id_buku'], [
            'stok' => $buku['stok'] - $input['jumlah_terjual'] // Mengurangi stok
        ]);

        return redirect()->to(base_url('pendapatan-buku'));
    }

    public function delete($id_penjualan)
{
    // Ambil data penjualan berdasarkan ID
    $penjualan = $this->penjualanBukuModel->find($id_penjualan);

    // Pastikan penjualan ditemukan
    if (is_null($penjualan)) {
        return redirect()->back()->with('error', 'Data penjualan tidak ditemukan.');
    }

    // Mengambil buku terkait untuk mengembalikan stok
    $buku = $this->bukuModel->find($penjualan['id_buku']);

    // Mengembalikan stok buku
    $this->bukuModel->update($penjualan['id_buku'], [
        'stok' => $buku['stok'] + $penjualan['jumlah_terjual'] // Mengembalikan stok
    ]);

    // Hapus data penjualan
    $this->penjualanBukuModel->delete($id_penjualan);

    return redirect()->to(base_url('pendapatan-buku'))->with('success', 'Penjualan berhasil dihapus dan stok dikembalikan.');
}
public function search()
{
    $start_date = $this->request->getGet('start_date');
    $end_date = $this->request->getGet('end_date');

    // Validasi input tanggal
    if (!$start_date || !$end_date) {
        return redirect()->to(base_url('pendapatan-buku'))->with('error', 'Tanggal harus diisi.');
    }

    // Ambil data penjualan berdasarkan rentang tanggal
    $data['penjualan'] = $this->penjualanBukuModel->getPenjualanByDateRange($start_date, $end_date);

    return view('dashboard/penjualan/index', $data);
}

public function exportToExcel()
{
    // Ambil data dari request (filter tanggal jika ada)
    $tanggal_mulai = $this->request->getGet('tanggal_mulai');
    $tanggal_selesai = $this->request->getGet('tanggal_selesai');

    // Jika tidak ada filter tanggal, set tanggal default
    if (!$tanggal_mulai) {
        $tanggal_mulai = date('Y-m-01'); // Awal bulan
    }
    if (!$tanggal_selesai) {
        $tanggal_selesai = date('Y-m-t'); // Akhir bulan
    }

    // Ambil data penjualan berdasarkan filter tanggal
    $penjualan = $this->penjualanBukuModel
        ->where('tanggal_penjualan >=', $tanggal_mulai)
        ->where('tanggal_penjualan <=', $tanggal_selesai)
        ->findAll();

    // Ambil semua ID buku dari penjualan
    $bukuIds = array_column($penjualan, 'id_buku');
    $bukuData = $this->bukuModel->whereIn('id_buku', $bukuIds)->findAll();

    // Buat array untuk memetakan ID buku ke nama buku
    $bukuMap = [];
    foreach ($bukuData as $buku) {
        $bukuMap[$buku['id_buku']] = $buku['nama_buku'];
    }

    // Gabungkan nama buku ke data penjualan
    foreach ($penjualan as &$item) {
        $item['nama_buku'] = $bukuMap[$item['id_buku']] ?? 'Tidak Ditemukan';
    }

    // Buat file Excel (misalnya menggunakan PhpSpreadsheet)
    // Inisialisasi spreadsheet dan set data
    $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();
    $sheet->setTitle('Laporan Penjualan');

    // Set header
    $sheet->setCellValue('A1', 'ID Penjualan');
    $sheet->setCellValue('B1', 'Nama Buku');
    $sheet->setCellValue('C1', 'Jumlah Terjual');
    $sheet->setCellValue('D1', 'Total Harga');
    $sheet->setCellValue('E1', 'Tanggal Penjualan');
    $sheet->setCellValue('F1', 'Keterangan');

    // Isi data
    $row = 2; // Mulai dari baris kedua
    foreach ($penjualan as $item) {
        $sheet->setCellValue('A' . $row, $item['id_penjualan']);
        $sheet->setCellValue('B' . $row, $item['nama_buku']);
        $sheet->setCellValue('C' . $row, $item['jumlah_terjual']);
        $sheet->setCellValue('D' . $row, $item['total_harga']);
        $sheet->setCellValue('E' . $row, $item['tanggal_penjualan']);
        $sheet->setCellValue('F' . $row, $item['keterangan']);
        $row++;
    }

    // Buat file dan kirim ke browser
    $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
    $filename = 'Laporan_Penjualan_' . date('YmdHis') . '.xlsx';

    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Cache-Control: max-age=0');

    $writer->save('php://output');
    exit;
}
public function exportToPDF()
{
    // Ambil data dari request (filter tanggal jika ada)
    $tanggal_mulai = $this->request->getGet('tanggal_mulai');
    $tanggal_selesai = $this->request->getGet('tanggal_selesai');

    // Jika tidak ada filter tanggal, set tanggal default
    if (!$tanggal_mulai) {
        $tanggal_mulai = date('Y-m-01'); // Awal bulan
    }
    if (!$tanggal_selesai) {
        $tanggal_selesai = date('Y-m-t'); // Akhir bulan
    }

    // Ambil data penjualan berdasarkan filter tanggal
    $penjualan = $this->penjualanBukuModel
        ->where('tanggal_penjualan >=', $tanggal_mulai)
        ->where('tanggal_penjualan <=', $tanggal_selesai)
        ->findAll();

    // Ambil semua ID buku dari penjualan
    $bukuIds = array_column($penjualan, 'id_buku');
    $bukuData = $this->bukuModel->whereIn('id_buku', $bukuIds)->findAll();

    // Buat array untuk memetakan ID buku ke nama buku
    $bukuMap = [];
    foreach ($bukuData as $buku) {
        $bukuMap[$buku['id_buku']] = $buku['nama_buku'];
    }

    // Gabungkan nama buku ke data penjualan
    foreach ($penjualan as &$item) {
        $item['nama_buku'] = $bukuMap[$item['id_buku']] ?? 'Tidak Ditemukan';
    }

    // Konversi ke PDF (misalnya menggunakan dompdf)
    $dompdf = new \Dompdf\Dompdf();
    $html = view('dashboard/penjualan/pdf', ['penjualan' => $penjualan]); // Pastikan Anda memiliki view yang sesuai
    $dompdf->loadHtml($html);
    $dompdf->setPaper('A4', 'landscape');
    $dompdf->render();

    // Kirim file PDF ke browser
    $filename = 'Laporan_Penjualan_' . date('YmdHis') . '.pdf';
    $dompdf->stream($filename, ['Attachment' => true]);
    exit;
}

}
